﻿

module("VOICE")

ASSERT(type(voice_debug_log) == "function", "Missing voice_debug_log function.")
ASSERT(type(voice_debug_log_table) == "function", "Missing voice_debug_log_table function.")

ASSERT(type(wjoin) == "function", "Missing wjoin function.")
ASSERT(type(wsplit) == "function", "Missing wsplit function.")
ASSERT(type(getFirstNChar) == "function", "Missing getFirstNChar function.")
ASSERT(type(getLastNChar) == "function", "Missing getLastNChar function.")

ASSERT(type(transform_and_format) == "function", "Missing transform_and_format function.")
ASSERT(type(transform_chain) == "function", "Missing transform_chain function.")
ASSERT(type(table_concat) == "function", "Missing table_concat function.")
ASSERT(type(transform_pattern_match) == "function", "Missing transform_pattern_match function.")

ASSERT(type(transform.roadname_abbrev_table) == "table", "Missing transform.roadname_abbrev_table table.")
ASSERT(type(transform.direction_abbrev) == "table", "Missing transform.direction_abbrev table.")


local replace_sentence={
	{L" +([.,])", L"%1"},
	{L" +", L" "},
	{L"[., ]+[Tt]hen", L", then"},
}

local destname = {
    EXITNAME    = L"at exit to %s",
    EXITNUMBER  = L"at exit %s",
    ROADNAME    = L"onto %s",
    DESTINATION = L"to %s",
    SETTLEMENT  = L"towards %s",
}

local number_formatter_2 = {
    
    { L"0(%d)", L"o %1" },
}

local number_formatter_3 = {
    
    { L"00(%d)", L"o o %1" },
    
    { L"0(%d%d)", L"o %1" },
    
    { L"([1-9])00", L"%1 hundred" },
    
    { L"([1-9])0([1-9])", L"%1 o %2" },
    
    { L"([1-9])([1-9]%d)", L"%1 %2" },
}

local number formatter_4 = {
    
    { L"([1-9])000", L"%1 thousand" },
    
    { L"([1-9])00([1-9])", L"%1 thousand %2" },
    
    { L"([1-9]0)([1-9]%d)", L"%1 %2" },
    
    { L"([1-9][1-9])00", L"%1 hundred" },
    
    { L"([1-9][1-9])0([1-9])", L"%1 o %2" },
    
    { L"([1-9][1-9])([1-9]%d)", L"%1 %2" },
}

local number_formatters = {
    [2] = number_formatter_2,
    [3] = number_formatter_3,
    [4] = number_formatter_4,
}

local function format_number(str)
    if type(str) ~= "wstring" then
        ASSERT(FALSE, "The input of format_number must be wstring. Type: "..type(str))
        return L""
    end
    local formatter = number_formatters[#str]
    if formatter then
        local rep
        for _,v in ipairs(formatter) do
            str, rep = wstring.gsub(str, v[1], v[2])
            if rep > 0 then break end
        end
    end
    return str
end

local roadnumber_typ_patterns = {
    L"^%d+",
    L"^%a%d+",
    L"^%a%a%d+",
}

local function is_road_number(str)
    return transform_pattern_match(str, roadnumber_typ_patterns)
end



local tail_replace_map = {
	{L"%(E%)", L"East"},
	{L"%(W%)", L"West"},
	{L"%(S%)", L"South"},
	{L"%(N%)", L"North"},
	{L"%(NE%)", L"North East"},
	{L"%(NW%)", L"North west"},
	{L"%(SE%)", L"South East"},
	{L"%(SW%)", L"South West"},
	{L"%(M%)", L"Motorway"},
}

local function format_road_number(roadnum)
    local res
    if is_road_number(roadnum) then
        roads = transform_roadnumber_explode_eu(roadnum)
        for k, str in ipairs(roads) do
            
            str = wstring.upper(str)
            
            str = wstring.gsub(str, L"(%a)-(%a)", L"%1 %2")
            _,_,head,num,tail = wstring.find(str, L"^(.-)(%d+)(.-)$")
            
            num = format_number(num)
            
            tail = transform.direction_abbrev:transform(tail)
            
            str = table_concat({head,num,tail}, L" ")
            
            str = wstring.gsub(str, L"%s%s+", L" ")
            
            roads[k] = str
        end
        res = table_concat(roads, L"; ")
    else
        res = transform.roadname_abbrev_table:transform(roadnum)
    end
    return res
end

local function format_road_name(str)
    if is_road_number(str) then
    
        str = format_road_number(str)
    else
    
        str = transform.roadname_abbrev_table:transform(str)
    end
    return str
end


local function format_signpost_exitnumber(str)
	local _, _, prefix, num, postfix = wstring.find(str, L"^(.-)(%d+)(.-)$")
	ASSERT(num, L"No number in format_signpost_exitnumber: "..str)
	if num then num = format_number(num) end
	
	if prefix then prefix = wstring.gsub(wstring.upper(prefix), L"EXIT%s*", L"") end
	
	if postfix then postfix = wstring.gsub(postfix, L"[/-]+", L", ") end
        
	return table_concat({prefix, num, postfix}, L" ")
end

local function signpost_exitnumber(data, idx)
    return transform_and_format(data[idx].signpost.exitnumber, format_signpost_exitnumber, destname.EXITNUMBER)
end

local function signpost_exitname(data, idx)
    return transform_and_format(data[idx].signpost.exitname, nil, destname.EXITNAME)
end

local function signpost_destination(data, idx)
    return transform_and_format(data[idx].signpost.destination, transform.roadname_abbrev_table,destname.DESTINATION)
end

local function signpost_settlement(data, idx)
        return transform_and_format(data[idx].signpost.settlement, transform.settlement_abbrev, destname.SETTLEMENT)
end

local function signpost_roadnumber(data, idx)
    local format
    if data[idx].signpost.sign_road_branch then format = destname.ROADNAME
    else format = destname.SETTLEMENT end

    return transform_and_format(data[idx].signpost.roadnumber, format_road_number, format)
end

local function road_name(data, idx)
    return transform_and_format(data[idx].road.name, format_road_name)
end

local function road_number(data, idx)
    return transform_and_format(data[idx].road.number, format_road_number)
end

function format_destname(data, idx)
    local t
    local out
    if data[idx].signpost then
        t = transform_chain({}, signpost_exitname, 	data, idx)
	t = transform_chain(t,  signpost_exitnumber, 	data, idx)
	t = transform_chain(t,  signpost_roadnumber, 	data, idx)
	t = transform_chain(t,  signpost_destination,	data, idx)
	t = transform_chain(t,  signpost_settlement, 	data, idx)
        out = table_concat(t, L", ")
    elseif data[idx].road then
	t = transform_chain({}, road_number, data, idx)
	t = transform_chain(t,  road_name,   data, idx)
        out = wstring.format(destname.ROADNAME, table_concat(t, L", "))
    end
    return out
end

function format_sentence(str)
	for key, value in ipairs(replace_sentence) do
		str = wstring.gsub(str, value[1], value[2])
	end
	return str
end

function format_streetname(data, idx)
    local t
	t = transform_chain({}, road_number, 	data, idx)
    t = transform_chain(t,  road_name, 	data, idx)
    return table_concat(t, L", ")
end

function route_summary_format_road_name(data)
        return transform_and_format(data, format_road_number)
end

function route_summary_format_street_name(data)
        return transform_and_format(data, transform.roadname_abbrev_table)
end

function route_summary_format_bridge_tunnel(data)
        return transform_and_format(data, transform.roadname_abbrev_table)
end

function route_summary_format_order(data)
        return transform_and_format(data, transform.roadname_abbrev_table)
end


function traffic_event_supported()
	return true
end

function traffic_event(DescKey, data)
	local str = translate_voice(DescKey)
	ASSERT(MODEL.regional.is_it_voice_localizable(DescKey), "Missing TrafficEvent dictionary.voice key:" .. DescKey)
	
	if data.roadnumber then 
		local road = transform_and_format(data.roadnumber, format_road_number, L" on %s")
		if data.from and not data.to then
		
			str =  str..road..L" near "..transform_and_format(data.from)
		elseif data.from and data.to then 
		
			str =  str..road..L", between "..transform_and_format(data.from)..L" and "..transform_and_format(data.to)
		else
		
			str = str..road
		end
	
	elseif data.roadname then
		local road = transform_and_format(data.roadname, transform.roadname_abbrev_table, L" on %s")
		if data.from and not data.to then
			
			local _,_, from, to = wstring.find(data.from.text, L"^([^,]+),(.+)$")
			if from and to then 
				str = str..road..L", at the intersection of "..from..L" and "..to
			else 
				str = str..road..L", at the intersection of "..transform_and_format(data.from)
			end
		elseif data.from and data.to then
			str = str..road..L", between "..transform_and_format(data.from)..L" and "..transform_and_format(data.to)
		else
			str = str..road
		end
	
	else
		if data.from and not data.to then
		
			str =  str..L" near "..transform_and_format(data.from)
		elseif data.from and data.to then 
		
			str =  str..L" between "..transform_and_format(data.from)..L" and "..transform_and_format(data.to)
		else
		
		end
	end
	str = str..L"."
	return str
end

function over_speed_limit()
	local key = m_i18n_voice("The speed limit is %s!")
	if MODEL.regional.is_it_voice_localizable(key) then
		
		local unitformat
		if announceSpeedUnit then
			unitformat = ESpeedDisplayPart.Full
		else
			unitformat = ESpeedDisplayPart.Number
		end
		local SpeedLimit = MODEL.warning.driveralert.speed_limit()
		local SpeedLimitStr = Format_Speed(SpeedLimit, ESpeedDisplayMode.Rounded, unitformat)
		SpeedLimitStr = transform_and_format(SpeedLimitStr, transform.measures_abbrev_table)
		local overspeedlimit = translated_voice_format(key, SpeedLimitStr)
		return overspeedlimit
	end
end
