﻿module("VOICE")

----------------------------------------------------------------------------------------------------
---------------------------|  V O I C E   T R A N S F O R M   U T I L S  |--------------------------
----------------------------------------------------------------------------------------------------

-----------------------------------------------
-- ASSERT a konyvtari transform fuggvenyekre --
-----------------------------------------------

assert(type(transform_and_format) == "function")
assert(type(transform_pattern_match) == "function")
assert(type(transform_chain) == "function")
assert(type(transform_format_roadnumber_eu) == "function")

local preposition_tbl = {L"auf den",L"auf das",L"auf die",L"in den",L"in die",L"in das",L"über die",L"richtung",L"auf die Straße "}

local preposition_insert_tbl = {
	{L"auf den ",{L"platz$",L"ring$",L"damm$",L"kanal$", L"^platz de[rs] "}},
	{L"auf das ",{L"dreieck$",L"kreuz$"}},
	{L"auf die ",{L"allee$",L"treppe$",L"^%a?%d+%a?$"}},
	{L"in den ",{L"horst$",L"kamp$",L"winkel$",L"pfad$",L"weg$",L"steg$",L"stieg$",L"hof$"}},
	{L"in die ",{L"gasse$",L"strase$",L"strasse$",L"sträse$",L"straße$",L"sträße$", L"^stra[ßs]+e de[rs] "}},
	{L"durch den ",{L"tunnel$",L"tunel$"}},
	{L"über die ",{L"brucke$",L"brücke$"}},
	{L"auf die Straße ",{L"berg$",L"park$",L"grund$",L"bahn$"}},
	{L"in das ",{L"tor$"}},
	{L"auf ",{L""}},
}

local traffic_preposition_insert_tbl = {
	-- in dem,der / von dem, der
	{L" dem ",{	L"platz$",L"ring$",L"damm$",L"berg$",L"kanal$", L"^platz de[rs] ", 
			L"dreieck$",L"kreuz$", L"horst$",L"kamp$",L"winkel$",L"pfad$",
			L"weg$",L"steg$",L"stieg$",L"hof$", L"tunnel$",L"tunel$", L"tor$"}},

	{L" der ",{	L"allee$",L"treppe$",L"^%a?%d+%a?$", L"gasse$",L"strase$",L"strasse$",
			L"sträse$",L"straße$",L"sträße$", L"^stra[ßs]+e de[rs] ", L"brucke$",L"brücke$"}},

	{L" von ",{L""}},
}

local settlement_preposition = L"richtung"
	
local replace_mapinfo = {
	{L"[sS][tT]%.",L"Sankt "},
	{L"^(%u)(%u)[ -]*(%d+)",L"%1.%2. %3"},
	{L"\-",L" "},
	{L"(%d)(%a)",L"%1 %2"},
	{L"(%a)(%d)",L"%1 %2"},
	{L"//",L", "},
}

local replace_sentence={
	{L"\. *[Dd]ann",L", dann"},
	{L"(%p)+ *(%p)+",L"%1 "},
	{L"%s+(%p+)",L"%1 "},
	{L" +",L" "},
}

local function get_preposition(str, tbl)
	str = wstring.lower(str)
	for k,value in ipairs(tbl) do
		for _,pattern in ipairs(tbl[k][2]) do
			if wstring.find(str,pattern) then
				return tbl[k][1] 
			end	
		end		
	end
end

----------------------------------------------------------------------------------------------------
----------------------------------------|  G U I D A N C E  |---------------------------------------
----------------------------------------------------------------------------------------------------

local function not_settlement(str)
	str = wstring.lower(str)
	if transform_pattern_match(str,preposition_tbl) then
		return L""
	end	
	return get_preposition(str, preposition_insert_tbl)
end

local function signpost_exitnumber(data, idx)
	local exitnumber = data[idx].signpost.exitnumber
	if exitnumber then 
		return not_settlement(exitnumber.text) .. transform_and_format(exitnumber,replace_mapinfo)
	end
end

local function signpost_exitname(data, idx)
	local exitname = data[idx].signpost.exitname
	if exitname then 
		return not_settlement(exitname.text) .. transform_and_format(exitname,replace_mapinfo)
	end
end

local function signpost_destination(data, idx)
	local dest = data[idx].signpost.destination
	if dest then
		return not_settlement(dest.text) .. transform_and_format(dest,replace_mapinfo)
	end
end

local function signpost_settlement(data, idx)
	local settlement = data[idx].signpost.settlement
	if settlement then
		local str = transform_and_format(settlement,replace_mapinfo)
		-- Ha van benn gyarilag "richtung"
		if wstring.find(settlement.text,settlement_preposition) then
			return str
		end
		-- Hozza kell tenni
		return settlement_preposition .. L" " .. str
	end
end

local function signpost_roadnumber(data, idx)
	-- Road numberhez nem hasznaljuk a fonemat
	local roadnumber = data[idx].signpost.roadnumber
	if roadnumber then 
		return not_settlement(roadnumber.text) .. transform_format_roadnumber_eu(roadnumber.text)
	end
end

local function road_name(data, idx)
	local name = data[idx].road.name
	if name then 
		return not_settlement(name.text) .. transform_and_format(name,replace_mapinfo)
	end
end

local function road_number(data, idx)
	local number = data[idx].road.number
	if number then 
		return not_settlement(number.text) .. transform_format_roadnumber_eu(number.text)
	end
end

function format_destname(data, idx)
	local t
	if data[idx].signpost then
		t = transform_chain({}, signpost_exitname, 	data, idx)
		t = transform_chain(t,  signpost_exitnumber, 	data, idx)
		t = transform_chain(t,  signpost_roadnumber, 	data, idx)
		t = transform_chain(t,  signpost_destination,	data, idx)
		t = transform_chain(t,  signpost_settlement, 	data, idx)
	elseif data[idx].road then
		t = transform_chain({}, road_number, 	data, idx)
		t = transform_chain(t,  road_name, 	data, idx)
	end
	return table_concat(t, L", ")
end

function format_streetname(data, idx)
	local t
	t = transform_chain({}, road_number, 	data, idx)
	t = transform_chain(t,  road_name, 	data, idx)
	return table_concat(t, L", ")
end

function format_sentence(str)
	for key, value in ipairs(replace_sentence) do
		str = wstring.gsub(str,value[1],value[2])
	end
	return str
end


----------------------------------------------------------------------------------------------------
-----------------------------------|  R O U T E   S U M M A R Y  |----------------------------------
----------------------------------------------------------------------------------------------------

function route_summary_format_road_name(data)
	if data.phoneme then return data.phoneme
	else return transform_format_roadnumber_eu(data.text)
	end
end

function route_summary_format_street_name(data)
	if data.phoneme then return data.phoneme
	else return data.text
	end
end

function route_summary_format_bridge_tunnel(data)
	if data.phoneme then return data.phoneme
	else return data.text
	end
end

function route_summary_format_order(data)
	if data.phoneme then return data.phoneme
	else return data.text
	end
end

----------------------------------------------------------------------------------------------------
-----------------------------------------|  T R A F F I C  |----------------------------------------
----------------------------------------------------------------------------------------------------

-- 1. Zar az M0 uton, Martonvasar kozeleben. = in der Strasse Nummer A1, in der Nähe von Martonvasar./ in der Nähe von der HaupStrasse Numer 33.
-- 2. Zar az M0 uton, Martonvasar es Velence kozt. = in der Strasse Nummer A1, zwischen Martonvasar und Velence.
-- 3. Zar a Budai uton, a Budai ut Fiskalis ut keresztezodesben = in der/dem (lsd tabla) Budai ut, in der Kreuzung Budai ut und Fiskalis ut.
-- 4. Zar a Budai uton, Fiskalis ut es 8. ut kozott. = in der/dem Budai ut, zwischen Fiskalis ut und 8. ut.

function traffic_event_supported()
	return true
end

function traffic_event(DescKey, data)
	local str = translate_voice(DescKey)
	local loc = MODEL.regional.is_it_voice_localizable(DescKey) 
	ASSERT(loc, "Missing TrafficEvent dictionary.voice key:" .. DescKey)
	---- Roadnumber prioritas
	if data.roadnumber then 
		local road = L" in der Strasse Nummer "..transform_format_roadnumber_eu(data.roadnumber.text)
		if data.from and not data.to then
		-- M7 uton Martonvasar kozeleben
			str =  str..road..L", in der Nähe von "..transform_and_format(data.from)
		elseif data.from and data.to then 
		-- M7 uton, Martonvasar es Velence kozott 
			str =  str..road..L", zwischen "..transform_and_format(data.from)..L" und "..transform_and_format(data.to)
		else
		-- Egyeb: M7 uton
			str = str..road
		end
	---- Roadname
	elseif data.roadname then
		-- Text/fonema + prepocicio
		local road = transform_and_format(data.roadname)
		road = L" in"..get_preposition(data.roadname.text, traffic_preposition_insert_tbl)..road
		if data.from and not data.to then
			-- "," menten robbantunk
			local _,_, from, to = wstring.find(data.from.text, L"^([^,]+),(.+)$")
			if from and to then 
				str = str..road..L", an der Kreuzung "..from..L" und "..to
			else 
				str = str..road..L", an der Kreuzung "..data.from.text
			end
		elseif data.from and data.to then
			str = str..road..L", zwischen "..transform_and_format(data.from)..L" und "..transform_and_format(data.to)
		else
			str = str..road
		end
	---- Nincs se roadnumber, se roadname 
	else
		if data.from and not data.to then
		-- Martonvasar kozeleben
			str =  str..L" in der Nähe von "..transform_and_format(data.from)
		elseif data.from and data.to then 
		-- Martonvasar es Velence kozott 
			str =  str..L" zwischen "..transform_and_format(data.from)..L" und "..transform_and_format(data.to)
		else
		-- Egyeb: semmi
		end
	end
	str = str..L"."
	return str
end

----------------------------------------------------------------------------------------------------
------------------------------------------|  D E T O U R  |-----------------------------------------
----------------------------------------------------------------------------------------------------

function format_street_name (streetname)
    if wstring.sub (streetname, 1, 9) == L "for_turns" then
        return transform_and_format (wstring.sub (streetname, 10), transform.roadname_abbrev_table, destname.ROADNAME)
    else
        return transform_and_format (streetname, transform.roadname_abbrev_table)
    end
end

