﻿module("VOICE")

ASSERT(type(voice_debug_log) == "function", "Missing voice_debug_log function.")
ASSERT(type(voice_debug_log_table) == "function", "Missing voice_debug_log_table function.")

ASSERT(type(wjoin) == "function", "Missing wjoin function.")
ASSERT(type(wsplit) == "function", "Missing wsplit function.")
ASSERT(type(getFirstNChar) == "function", "Missing getFirstNChar function.")
ASSERT(type(getLastNChar) == "function", "Missing getLastNChar function.")

ASSERT(type(transform_and_format) == "function", "Missing transform_and_format function.")
ASSERT(type(transform_chain) == "function", "Missing transform_chain function.")
ASSERT(type(table_concat) == "function", "Missing table_concat function.")
ASSERT(type(transform_pattern_match) == "function", "Missing transform_pattern_match function.")
ASSERT(type(transform_format_roadnumber_eu) == "function", "Missing transform_format_roadnumber_eu function.")

local prepositions = {
	L"auf das",
	L"auf den",
	L"auf die",
	L"in das",
	L"in den",
	L"in die",
	L"durch den",
	L"über die",
	L"richtung",
	L"auf ",
}

local preposition_accusatives = {
	{L"auf das ", {L"dreieck$", L"dreieck +%a+", L"kreuz$", L"kreuz +%a+"}},
	{L"auf den ", {L"platz$", L"ring$", L"damm$", L"berg$", L"kanal$", L"^platz de[rs] "}},
	{L"auf die ", {L"allee$", L"treppe$", L"steige$", L"anschlussstelle", L"^%a?%d+%a?$"}},
	{L"in das ", {L"tor$"}},
	{L"in den ", {L"horst$", L"kamp$", L"winkel$", L"pfad$", L"weg$", L"steg$", L"stieg$", L"hof$"}},
	{L"in die ", {L"gasse$", L"strase$", L"strasse$", L"sträse$", L"straße$", L"sträße$", L"^stra[ßs]+e de[rs] "}},
	{L"durch den ", {L"tunnel$", L"tunel$"}},
	{L"über die ", {L"brucke$", L"brücke$"}},
	{L"auf ", {L""}},
}

local traffic_preposition_insert_tbl = {
    {L"auf dem ", {L"platz$", L"ring$", L"damm$", L"berg$", L"kanal$", L"^platz de[rs] ", L"dreieck$", L"kreuz$"}},
    {L"auf der ", {L"allee$", L"treppe$", L"steige$", L"^%a?%d+%a?$", L"brucke$", L"brücke$"}},
    {L"im ", {L"horst$", L"kamp$", L"winkel$", L"pfad$", L"weg$", L"steg$", L"stieg$", L"hof$", L"tunnel$", L"tunel$", L"tor$"}},
    {L"in der ", {L"gasse$", L"strase$", L"strasse$", L"sträse$", L"straße$", L"sträße$", L"^stra[ßs]+e de[rs] "}},
    {L"auf ", {L""}},
} 

local traffic_preposition_from_tbl = {
    {L"am ", {L"platz$", L"ring$", L"damm$", L"berg$", L"kanal$", L"^platz de[rs] ", L"dreieck$", L"kreuz$"}},
    {L"an der ", {L"allee$", L"treppe$", L"steige$", L"^%a?%d+%a?$", L"brucke$", L"brücke$"}},
    {L"am ", {L"horst$", L"kamp$", L"winkel$", L"pfad$", L"weg$", L"steg$", L"stieg$", L"hof$", L"tunnel$", L"tunel$", L"tor$"}},
    {L"an der ", {L"gasse$", L"strase$", L"strasse$", L"sträse$", L"straße$", L"sträße$", L"^stra[ßs]+e de[rs] "}},
    {L"bei ", {L""}},
} 
	
local replace_mapinfo = {
	{L"[sS][tT]%.", L"Sankt "},
	{L"^(%u)(%u)[ -]*(%d+)", L"%1.%2. %3"},
	{L"\-", L" "},
	{L"(%d)(%a)", L"%1 %2"},
	{L"(%a)(%d)", L"%1 %2"},
	{L"//", L", "},
}
	
local replace_exitnumber = {
	{L"(%d+)(%a)$", L"%1 %2"},
	{L"(%d+)$", L"%1"},
	{L"(%a)(%d+)$", L"%1 %2"},
}

local replace_sentence={
	{L"%. *[Dd]ann", L", dann"},
	{L"%. *[Dd]anach", L", danach"},
	{L"%. *Sie kommen dann", L", Sie kommen dann"},
	{L"(%p)+ *(%p)+", L"%1 "},
	{L"%s+(%p+)", L"%1 "},
	{L" +", L" "},
	{L"(%d+)(%.)", L"%1 %2"},
}

local function find_preposition(str, tbl)
	str = wstring.lower(str)
	for k, value in ipairs(tbl) do
		for _, pattern in ipairs(tbl[k][2]) do
			if wstring.find(str, pattern) then
				return tbl[k][1] 
			end	
		end		
	end
	return L""
end

local function get_pre_post_position(data)
	
	local prepos
	local str = wstring.lower(data.text)
	local _, found_preposition = transform_pattern_match(str, prepositions)
	if found_preposition then
		prepos = L""
	else
		prepos = find_preposition(str, preposition_accusatives)
	end

	
	local postpos = L""

	return prepos, postpos
end


local function signpost_exitnumber(data, idx)
	local exitnumber = data[idx].signpost.exitnumber
	if exitnumber then 
		return L"an der Ausfahrt " .. transform_and_format(exitnumber, replace_exitnumber)
	end
end

local function signpost_exitname(data, idx)
	local exitname = data[idx].signpost.exitname
	if exitname then 
		local prepos, postpos = get_pre_post_position(exitname)
		return prepos .. transform_and_format(exitname, replace_mapinfo) .. postpos
	end
end

local function signpost_destination(data, idx)
	local dest = data[idx].signpost.destination
	if dest then
		local prepos, postpos = get_pre_post_position(dest)
		return prepos .. transform_and_format(dest, replace_mapinfo) .. postpos
	end
end

local function signpost_settlement(data, idx)
	local settlement = data[idx].signpost.settlement
	if settlement then
		local settlement_preposition = L"richtung"
		local str = transform_and_format(settlement, replace_mapinfo)
		
		if wstring.find(settlement.text, settlement_preposition) then
			return str
		else
			return settlement_preposition .. L" " .. str
		end
	end
end

local function signpost_roadnumber(data, idx)
	
	local roadnumber = data[idx].signpost.roadnumber
	if roadnumber then
		local prepos, postpos = get_pre_post_position(roadnumber)
		return prepos .. transform_format_roadnumber_eu(roadnumber.text) .. postpos
	end
end

local function road_name(data, idx)
	local name = data[idx].road.name
	if name then
		local prepos, postpos = get_pre_post_position(name)
		return prepos .. transform_and_format(name, replace_mapinfo) .. postpos
	end
end

local function road_number(data, idx)
	local number = data[idx].road.number
	if number then
		local prepos, postpos = get_pre_post_position(number)
		return prepos .. transform_format_roadnumber_eu(number.text) .. postpos
	end
end

function format_destname(data, idx)
	local t
	if data[idx].signpost then
		t = transform_chain({}, signpost_exitname, 	data, idx)
		t = transform_chain(t,  signpost_exitnumber, 	data, idx)
		t = transform_chain(t,  signpost_roadnumber, 	data, idx)
		t = transform_chain(t,  signpost_destination,	data, idx)
		t = transform_chain(t,  signpost_settlement, 	data, idx)
	elseif data[idx].road then
		t = transform_chain({}, road_number, 	data, idx)
		t = transform_chain(t,  road_name, 	data, idx)
	end
	return table_concat(t, L", ")
end

function format_streetname(data, idx)
	local t
	t = transform_chain({}, road_number, 	data, idx)
	t = transform_chain(t,  road_name, 	data, idx)
	return table_concat(t, L", ")
end

function format_sentence(str)
	for key, value in ipairs(replace_sentence) do
		str = wstring.gsub(str, value[1], value[2])
	end
	return str
end



function route_summary_format_road_name(data)
	if data.phoneme then return data.phoneme
	else return transform_format_roadnumber_eu(data.text)
	end
end

function route_summary_format_street_name(data)
	if data.phoneme then return data.phoneme
	else return data.text
	end
end

function route_summary_format_bridge_tunnel(data)
	if data.phoneme then return data.phoneme
	else return data.text
	end
end

function route_summary_format_order(data)
	if data.phoneme then return data.phoneme
	else return data.text
	end
end



function traffic_event_supported()
	return true
end

function traffic_event(DescKey, data)
	local str = translate_voice(DescKey)
	local loc = MODEL.regional.is_it_voice_localizable(DescKey) 
	ASSERT(loc, "Missing TrafficEvent dictionary.voice key:" .. DescKey)
	voice_debug_log(L"Desc: " .. str)
	
	local road=L""
	
	if data.roadnumber and data.roadnumber.text and data.country and data.country.text==L"_GER" then
		ASSERT(data.roadnumber.text:len())
		if data.roadnumber.text:len() and data.roadnumber.text:sub(1, 1)~=L"A" and  data.roadnumber.text:sub(1, 1)~=L"B" then
			data.roadnumber=nil
		end
	end
	
	

	if data.roadnumber then
		road = L" auf der " .. transform_format_roadnumber_eu(data.roadnumber.text)
	
	elseif data.roadname then
		road = transform_and_format(data.roadname)
		road = find_preposition(data.roadname.text, traffic_preposition_insert_tbl) .. road
	end
	str = str .. road
	
	if data.from and not data.to then
		local from =data.from.text
		local to=L""
		if not data.to then
		
		local _, _, f, t = wstring.find(data.from.text, L"^([^,]+),(.+)$")
			if f then 
				from=f
				to=t
			end
		end
		
		if from and to == L"" then
			str =  str .. L" , bei " .. transform_and_format(from)
		else
			str =  str .. L" , zwischen " .. transform_and_format(from) .. L" und " .. transform_and_format(to)
		end	
	elseif data.from and data.to then 
		
			str =  str .. L" , zwischen " .. transform_and_format(data.from) .. L" und " .. transform_and_format(data.to)
	
	end
	str = str .. L"."
	voice_debug_log(L"Sentence to say: " .. str)
	return str
end

